<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Pdf;
use Illuminate\Support\Facades\Storage;

class PdfController extends Controller
{
    // Show upload form
    public function create()
    {
        return view('pdfs.create');
    }

    // Handle upload
public function store(Request $request)
{
    // Validate input
    $request->validate([
        'title' => 'required|string|max:255',
        'pdf_file' => 'nullable|mimes:pdf|max:10240', // optional PDF, max 10MB
    ]);

    try {
        $pdfFilePath = null;

        // Ensure PDF folder exists
        $pdfFolder = public_path('uploads/pdfs');
        if (!file_exists($pdfFolder)) {
            mkdir($pdfFolder, 0777, true);
        }

        // Upload PDF if exists
        if ($request->hasFile('pdf_file')) {
            $pdf = $request->file('pdf_file');
            $pdfName = time() . '_' . $pdf->getClientOriginalName();
            $pdf->move($pdfFolder, $pdfName);
            $pdfFilePath = 'uploads/pdfs/' . $pdfName;
        }

        // Save to database
        Pdf::create([
            'title' => $request->title,
            'file_path' => $pdfFilePath,
        ]);

        return redirect()->route('pdfs.index')->with('success', 'PDF uploaded successfully!');
    } catch (\Exception $e) {
        return redirect()->back()->with('error', 'Something went wrong: ' . $e->getMessage())->withInput();
    }
}

public function update(Request $request, Pdf $pdf)
{
    // Validate input
    $request->validate([
        'title' => 'required|string|max:255',
        'pdf_file' => 'nullable|mimes:pdf|max:10240',
    ]);

    try {
        $pdfFilePath = $pdf->file_path;

        // Upload new PDF if exists
        if ($request->hasFile('pdf_file')) {
            $pdfFolder = public_path('uploads/pdfs');
            if (!file_exists($pdfFolder)) {
                mkdir($pdfFolder, 0777, true);
            }

            $pdfFile = $request->file('pdf_file');
            $pdfName = time() . '_' . $pdfFile->getClientOriginalName();
            $pdfFile->move($pdfFolder, $pdfName);

            $pdfFilePath = 'uploads/pdfs/' . $pdfName;
        }

        // Update PDF record
        $pdf->update([
            'title' => $request->title,
            'file_path' => $pdfFilePath,
        ]);

        return redirect()->back()->with('success', 'PDF updated successfully!');
    } catch (\Exception $e) {
        return redirect()->back()->with('error', 'Failed to update PDF: ' . $e->getMessage());
    }
}




public function destroy(Pdf $pdf)
{
    try {
        // Delete PDF file from storage
        if (Storage::disk('public')->exists($pdf->file_path)) {
            Storage::disk('public')->delete($pdf->file_path);
        }

        // Delete database record
        $pdf->delete();

        return redirect()->back()->with('success', 'PDF deleted successfully!');
    } catch (\Exception $e) {
        return redirect()->back()->with('error', 'Failed to delete PDF: ' . $e->getMessage());
    }
}


    // Show all PDFs
    public function index()
    {
        $pdfs = Pdf::all();
        return view('admin.pdflist', compact('pdfs'));
    }

    // Download PDF
    public function download(Pdf $pdf)
    {
        return Storage::disk('public')->download($pdf->file_path);
    }
    public function downloads()
    {
        $pdfs = Pdf::latest()->get(); // Fetch all PDFs, latest first
        return view('downloads', compact('pdfs')); // Pass to view
    }
}
